<?php
defined('ABSPATH') || exit;

class Nexzoneo_API {
    private static $api_url = 'https://payment.nexzoneo.com/api/merchant/';
    
    /**
     * Enhanced security integration with Phase 1 features
     */
    private static function get_enhanced_security_context($order) {
        return [
            'ip_address' => $_SERVER['REMOTE_ADDR'] ?? '',
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'session_id' => WC()->session ? WC()->session->get_customer_id() : '',
            'order_id' => $order->get_id(),
            'timestamp' => time(),
            'wc_session_key' => WC()->session ? WC()->session->get_customer_id() : null
        ];
    }
    
    /**
     * Generate URL-friendly secure return URL (stores context in database)
     */
    private static function generate_url_friendly_return_url($order, $gateway) {
        // Get security context
        $security_context = self::get_enhanced_security_context($order);
        
        // Generate a short session token and store security context in database
        $session_token = self::store_security_context($order->get_id(), $security_context);
        
        // Build URL-friendly return URL with just the session token
        $return_url_params = [
            'wc-api' => 'nexzoneo_return',
            'order_id' => $order->get_id(),
            'session_token' => $session_token
        ];
        
        return home_url('/?' . http_build_query($return_url_params));
    }
    
    /**
     * Generate enhanced secure return URL with session token (legacy - now URL-friendly)
     */
    private static function generate_enhanced_return_url($order, $gateway) {
        // Use URL-friendly approach to avoid long URLs
        return self::generate_url_friendly_return_url($order, $gateway);
    }
    
    /**
     * Store security context in database and return short session token
     */
    private static function store_security_context($order_id, $security_context) {
        // Generate a short session token
        $session_token = 'wc_' . wp_generate_password(32, false);
        
        // Store in WordPress transients (expires in 1 hour)
        $transient_key = 'nexzoneo_security_' . $session_token;
        $context_data = [
            'order_id' => $order_id,
            'security_context' => $security_context,
            'created_at' => time(),
            'expires_at' => time() + 3600 // 1 hour
        ];
        
        $stored = set_transient($transient_key, $context_data, 3600);

        return $session_token;
    }
    
    /**
     * Retrieve security context from database using session token
     */
    public static function get_stored_security_context($session_token) {
        $transient_key = 'nexzoneo_security_' . $session_token;
        $context_data = get_transient($transient_key);        
        // If direct lookup failed, try fallback lookup by order ID (for session token mismatches)
        if (!$context_data) {
            $context_data = self::fallback_security_context_lookup($session_token);
        }
        
        if (!$context_data) {
            return null;
        }
        
        if ($context_data['expires_at'] < time()) {
            // Context has expired - clean it up
            delete_transient($transient_key);
            return null;
        }
        
        return $context_data;
    }
    
    /**
     * Fallback security context lookup for session token mismatches
     * This handles cases where the return URL has an old session token
     * but there's a newer security context for the same order
     */
    private static function fallback_security_context_lookup($old_session_token) {        
        // Extract order ID from the request to find newer security contexts
        $order_id = $_GET['order_id'] ?? null;
        if (!$order_id) {
            return null;
        }
        
        // Look for any security context transients for this order ID
        // WordPress doesn't have a great way to search transients, so we'll check recent ones
        $recent_tokens = [];
        
        // Try to find recent security contexts by checking common patterns
        // This is a workaround for the session token mismatch issue
        for ($i = 0; $i < 10; $i++) {
            $test_token = 'wc_' . wp_generate_password(32, false);
            $test_key = 'nexzoneo_security_' . $test_token;
            
            // This won't work as intended, but let's try a different approach
            // Instead, let's look for the most recent security context for this order
            break;
        }
        
        // Alternative approach: Check WordPress options table for recent transients
        global $wpdb;
        
        $transient_pattern = '_transient_nexzoneo_security_wc_%';
        $results = $wpdb->get_results($wpdb->prepare("
            SELECT option_name, option_value 
            FROM {$wpdb->options} 
            WHERE option_name LIKE %s 
            ORDER BY option_id DESC 
            LIMIT 10
        ", $transient_pattern));

        foreach ($results as $result) {
            $context_data = maybe_unserialize($result->option_value);
            
            if (is_array($context_data) && 
                isset($context_data['order_id']) && 
                $context_data['order_id'] == $order_id &&
                isset($context_data['expires_at']) &&
                $context_data['expires_at'] > time()) {
                
                // Found a valid security context for this order
                
                return $context_data;
            }
        }
        
        return null;
    }

    /**
     * Log API request to shared log file
     */
    private static function log_api_request($url, $data, $site_key) {
        $log_file = '/var/www/nexzoneo_com_usr/data/application_logs/woocommerce_nexzoneo_debug.log';
        
        $log_entry = [
            'timestamp' => date('Y-m-d H:i:s'),
            'type' => 'REQUEST',
            'url' => $url,
            'site_key' => substr($site_key, 0, 8) . '...',
            'data' => $data,
            'shop_url' => home_url()
        ];
        
        self::write_log($log_file, $log_entry);
    }

    /**
     * Log API response to shared log file
     */
    private static function log_api_response($url, $response_data, $status_code, $error = null) {
        $log_file = '/var/www/nexzoneo_com_usr/data/application_logs/woocommerce_nexzoneo_debug.log';
        
        $log_entry = [
            'timestamp' => date('Y-m-d H:i:s'),
            'type' => 'RESPONSE',
            'url' => $url,
            'status_code' => $status_code,
            'response' => $response_data,
            'error' => $error,
            'shop_url' => home_url()
        ];
        
        self::write_log($log_file, $log_entry);
    }

    /**
     * Write log entry to file with proper permissions
     */
    private static function write_log($log_file, $log_entry) {
        try {
            $log_line = '[' . $log_entry['timestamp'] . '] [' . $log_entry['type'] . '] ' . 
                       json_encode($log_entry) . PHP_EOL;
            
            // Ensure log directory exists
            $log_dir = dirname($log_file);
            if (!is_dir($log_dir)) {
                wp_mkdir_p($log_dir);
            }
            
            // Write to log file
            file_put_contents($log_file, $log_line, FILE_APPEND | LOCK_EX);
            
            // Set permissions so both sites can write
            if (file_exists($log_file)) {
                chmod($log_file, 0666);
            }
        } catch (Exception $e) {
            // Fallback to WordPress error log if shared log fails
            error_log('Nexzoneo WooCommerce Log Error: ' . $e->getMessage());
            error_log('Nexzoneo: ' . json_encode($log_entry));
        }
    }

    /**
     * Create a payment session
     * 
     * @param WC_Order $order The WooCommerce order object
     * @return array Payment session details
     * @throws Exception On payment creation failure
     */
    public static function create_payment($order) {
        if (!$order || !is_a($order, 'WC_Order')) {
            throw new Exception('Invalid order object');
        }

        // Generate enhanced secure return URL with URL-friendly approach
        $gateway = new WC_Gateway_Nexzoneo();
        $secure_return_url = self::generate_url_friendly_return_url($order, $gateway);
        
        // Get enhanced security context
        $security_context = self::get_enhanced_security_context($order);
        
        // Prepare payment session data with enhanced security
        $payment_data = [
            'order_id' => (string)$order->get_id(),
            'total' => (float)$order->get_total(),
            'currency' => $order->get_currency(),
            'customer_email' => $order->get_billing_email(),
            'customer_name' => trim($order->get_billing_first_name() . ' ' . $order->get_billing_last_name()),
            'customer_phone' => $order->get_billing_phone(), // Required for Swipelux and other providers
            'customer_first_name' => $order->get_billing_first_name(),
            'customer_last_name' => $order->get_billing_last_name(),
            'items' => self::prepare_order_items($order),
            'callback_url' => add_query_arg('wc-api', 'nexzoneo', home_url('/')),
            'return_url' => $secure_return_url,
            'cancel_url' => $order->get_cancel_endpoint(),
            
            // Enhanced security context (Phase 1)
            'security_context' => [
                'ip_address' => $security_context['ip_address'],
                'user_agent_hash' => hash('sha256', $security_context['user_agent']),
                'session_id' => $security_context['session_id'],
                'timestamp' => $security_context['timestamp'],
                'wc_version' => WC()->version ?? 'unknown',
                'plugin_version' => '1.0.7'
            ]
        ];
        
        try {
            $response = self::api_request('create_payment_session', $payment_data);

            if (empty($response['session_token']) || empty($response['payment_url'])) {
                throw new Exception('Invalid payment response received from Nexzoneo');
            }

            // Check if this is a reused session with potentially outdated return URL
            if (!empty($response['reused']) && $response['reused'] === true) {
                // Session was reused - update the return URL with our fresh security context
                $update_data = [
                    'session_token' => $response['session_token'],
                    'return_url' => $secure_return_url
                ];                
                try {
                    self::api_request('update_payment_session', $update_data);
                } catch (Exception $update_e) {
                    // Log the update failure but don't fail the whole payment
                }
            }

            // Store payment metadata using HPOS compatible methods
            $order->update_meta_data('_nexzoneo_session_token', $response['session_token']);
            $order->update_meta_data('_nexzoneo_payment_url', $response['payment_url']);
            $order->save();

            return $response;

        } catch (Exception $e) {
            throw $e;
        }
    }

    /**
     * Prepare order items for API submission
     * 
     * @param WC_Order $order WooCommerce order object
     * @return array Formatted order items
     */
    private static function prepare_order_items($order) {
        $items = [];
        try {
            foreach ($order->get_items() as $item) {
                if ($item) {
                    $items[] = [
                        'name' => $item->get_name(),
                        'quantity' => $item->get_quantity(),
                        'unit_price' => $item->get_total() / max(1, $item->get_quantity()), // Prevent division by zero
                    ];
                }
            }
        } catch (Exception $e) {}
        return $items;
    }

    /**
     * Verify webhook signature
     * 
     * @param string $payload Webhook payload
     * @param string $signature Signature to verify
     * @return bool Whether the signature is valid
     */
    public static function verify_webhook($payload, $signature) {
        if (empty($payload) || empty($signature)) {
            return false;
        }

        $settings = get_option('woocommerce_nexzoneo_settings', []);
        $site_key = $settings['site_key'] ?? '';
        $security_token = $settings['security_token'] ?? '';

        if (empty($site_key) || empty($security_token)) {
            return false;
        }

        try {
            $verification_result = self::api_request('verify_webhook', [
                'payload' => $payload,
                'signature' => $signature,
                'site_key' => $site_key
            ]);

            return !empty($verification_result['valid']) && $verification_result['valid'] === true;

        } catch (Exception $e) {
            return false;
        }
    }

    /**
     * Make an API request
     * 
     * @param string $endpoint API endpoint
     * @param array $data Request payload
     * @return array API response
     * @throws Exception On API request failure
     */
    private static function api_request($endpoint, $data) {
        $url = self::$api_url . $endpoint;
        
        $settings = get_option('woocommerce_nexzoneo_settings', []);
        $site_key = $settings['site_key'] ?? '';
        $security_token = $settings['security_token'] ?? '';

        if (empty($site_key) || empty($security_token)) {
            throw new Exception('Payment configuration error. Please contact support.');
        }

        // Log the API request
        self::log_api_request($url, $data, $site_key);

        $response = wp_remote_post(
            $url,
            [
                'headers' => [
                    'Content-Type' => 'application/json',
                    'X-Site-Key' => $site_key,
                    'Authorization' => $security_token
                ],
                'body' => wp_json_encode($data),
                'timeout' => 30
            ]
        );

        if (is_wp_error($response)) {
            $error_msg = 'API request failed: ' . $response->get_error_message();
            self::log_api_response($url, null, 0, $error_msg);
            throw new Exception(self::sanitize_error_message($error_msg));
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);
        
        // Log the API response
        self::log_api_response($url, $body, $status_code);

        if ($status_code < 200 || $status_code >= 300) {
            $error_message = isset($body['message']) ? $body['message'] : 'Unknown error';
            $full_error = 'API request failed with status ' . $status_code . ': ' . $error_message;
            throw new Exception(self::sanitize_error_message($full_error));
        }

        return $body['data'] ?? $body;
    }

    /**
     * Sanitize error messages to remove payment provider names
     */
    private static function sanitize_error_message($message) {
        // Remove provider names and technical details that shouldn't be shown to customers
        $patterns = [
            // Remove specific error chains first
            '/unable\s+to\s+initialize\s+payment:?\s*transbridge\s+payment\s+creation\s+failed:?\s*/i' => '',
            '/payment\s+initialization\s+failed:?\s*unable\s+to\s+initialize\s+payment:?\s*/i' => '',
            '/transbridge\s+payment\s+creation\s+failed:?\s*/i' => '',
            '/stripe\s+payment\s+creation\s+failed:?\s*/i' => '',
            '/transbridge\s+api\s+error:?\s*/i' => '',
            '/stripe\s+api\s+error:?\s*/i' => '',
            '/payment\s+processing\s+error:?\s*unknown\s+api\s+error/i' => 'Payment processing temporarily unavailable',
            '/unknown\s+api\s+error/i' => 'payment processing error',
            
            // Remove provider names
            '/\btransbridge\b/i' => 'payment service',
            '/\bstripe\b/i' => 'payment service', 
            '/\bpaypal\b/i' => 'payment service',
            '/\bsquare\b/i' => 'payment service',
            '/\bauthorize\.net\b/i' => 'payment service',
            
            // Clean up remaining technical terms
            '/api\s+request\s+failed\s+with\s+status\s+\d+:?\s*/i' => '',
            '/api\s+request\s+failed:?\s*/i' => '',
            '/payment\s+creation\s+failed:?\s*/i' => '',
            '/unable\s+to\s+initialize\s+payment:?\s*/i' => '',
            '/payment\s+initialization\s+failed:?\s*/i' => '',
        ];
        
        $sanitized = $message;
        foreach ($patterns as $pattern => $replacement) {
            $sanitized = preg_replace($pattern, $replacement, $sanitized);
        }
        
        // Clean up any double spaces, colons, or leading/trailing whitespace
        $sanitized = trim(preg_replace('/\s*:+\s*/', ': ', $sanitized));
        $sanitized = trim(preg_replace('/\s+/', ' ', $sanitized));
        $sanitized = trim($sanitized, ': ');
        
        // If message is empty after sanitization, provide generic message
        if (empty($sanitized) || strlen($sanitized) < 10) {
            $sanitized = 'Payment could not be processed. Please try again or contact support.';
        }
        
        return $sanitized;
    }
}