<?php
defined('ABSPATH') || exit;

class WC_Gateway_Nexzoneo extends WC_Payment_Gateway {
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->id = 'nexzoneo';
        $this->method_title = 'Nexzoneo Payment';
        $this->method_description = 'Accept payments through Nexzoneo payment gateway with enhanced reliability';
        $this->supports = [
            'products',
            'refunds',
            'subscriptions',
        ];
        
        $this->init_form_fields();
        $this->init_settings();

        $this->enabled = $this->get_option('enabled');
        $this->title = $this->get_option('title');
        $this->description = $this->get_option('description');
        $this->site_key = $this->get_option('site_key');
        $this->security_token = $this->get_option('security_token');
        $this->webhook_secret = $this->get_option('webhook_secret');
        $this->environment = $this->get_option('environment', 'production');
        $this->debug = $this->get_option('debug', 'no');

        add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
        add_action('woocommerce_api_nexzoneo_return', array($this, 'handle_return'));
        add_action('woocommerce_api_nexzoneo', array($this, 'handle_webhook'));
        
        // Enhanced webhook handling with multiple endpoints
        add_action('woocommerce_api_nexzoneo_webhook', array($this, 'handle_webhook'));
        add_action('woocommerce_api_nexzoneo_callback', array($this, 'handle_webhook'));
    }

    /**
     * Initialize form fields
     */
    public function init_form_fields() {
        $this->form_fields = array(
            'enabled' => array(
                'title' => 'Enable/Disable',
                'type' => 'checkbox',
                'label' => 'Enable Nexzoneo Pay',
                'default' => 'no'
            ),
            'title' => array(
                'title' => 'Title',
                'type' => 'text',
                'description' => 'Payment method title that customers see during checkout.',
                'default' => 'Credit Card / Debit Card',
                'desc_tip' => true
            ),
            'description' => array(
                'title' => 'Description',
                'type' => 'textarea',
                'description' => 'Payment method description that customers see during checkout.',
                'default' => 'Pay securely with your credit card, debit card, or bank transfer via Nexzoneo',
                'desc_tip' => true
            ),
            'environment' => array(
                'title' => 'Environment',
                'type' => 'select',
                'description' => 'Select the environment for processing payments.',
                'default' => 'production',
                'desc_tip' => true,
                'options' => array(
                    'production' => 'Production',
                    'sandbox' => 'Sandbox (Testing)'
                )
            ),
            'site_key' => array(
                'title' => 'Site Key',
                'type' => 'password',
                'description' => 'Get your Site Key from your Nexzoneo merchant dashboard.',
                'desc_tip' => true
            ),
            'security_token' => array(
                'title' => 'Security Token',
                'type' => 'password',
                'description' => 'Get your Security Token from your Nexzoneo merchant dashboard.',
                'desc_tip' => true
            ),
            'webhook_secret' => array(
                'title' => 'Webhook Secret',
                'type' => 'password',
                'description' => 'Get your Webhook Secret from your Nexzoneo merchant dashboard for secure webhook validation.',
                'desc_tip' => true
            ),
            'webhook_info' => array(
                'title' => 'Webhook URL',
                'type' => 'title',
                'description' => 'Configure this URL in your Nexzoneo dashboard: <code>' . home_url('/wc-api/nexzoneo') . '</code><br/>
                                 Alternative URLs: <code>' . home_url('/wc-api/nexzoneo_webhook') . '</code> or <code>' . home_url('/wc-api/nexzoneo_callback') . '</code>',
            ),
            'debug' => array(
                'title' => 'Debug Mode',
                'type' => 'checkbox',
                'label' => 'Enable debug logging',
                'description' => 'Log payment events for debugging purposes.',
                'default' => 'no',
                'desc_tip' => true
            )
        );
    }

    /**
     * Process payment
     */
    public function process_payment($order_id) {
        try {
            $order = wc_get_order($order_id);
            if (!$order) {
                throw new Exception('Order not found');
            }

            $order->update_status('pending', __('Awaiting Nexzoneo payment', 'woocommerce-nexzoneo'));
            
            // Store security metadata for validation
            $order->update_meta_data('_payment_ip', $_SERVER['REMOTE_ADDR'] ?? '');
            $order->update_meta_data('_payment_user_agent', $_SERVER['HTTP_USER_AGENT'] ?? '');
            $order->update_meta_data('_payment_timestamp', time());
            
            // Store session information for validation
            if (WC()->session) {
                $order->update_meta_data('_payment_session_id', WC()->session->get_customer_id());
                // Store customer email in session for guest validation
                WC()->session->set('customer_email', $order->get_billing_email());
            }
            
            $response = Nexzoneo_API::create_payment($order);
            
            if (empty($response['payment_url'])) {
                throw new Exception('Payment URL not received from API');
            }

            $order->update_meta_data('_nexzoneo_payment_initiated', true);
            $order->save();

            WC()->cart->empty_cart();

                return array(
                    'result' => 'success',
                    'redirect' => $response['payment_url']
                );
            
        } catch (Exception $e) {
            wc_add_notice('Payment error: ' . $e->getMessage(), 'error');
            return array('result' => 'fail');
        }
    }

    /**
     * Handle return from payment gateway
     */
    public function handle_return() {
        // Handle both POST (with signature) and GET (TransBridge redirect) requests
        $data = $_POST;
        $signature = $data['signature'] ?? '';
        
        // If no POST data or signature, handle as TransBridge GET redirect
        if (empty($data) || empty($signature)) {
            $this->handle_transbridge_return();
            return;
        }
        
        $verify_data = $this->prepare_verify_data($data);
        
        if (!$this->verify_signature($verify_data, $signature)) {
            // Critical security error - show custom error page
            $this->show_payment_error('Invalid payment signature. Please contact support if you were charged.');
            exit;
        }
    
        $order = wc_get_order($data['order_id'] ?? '');
        if (!$order) {
            // Critical error - show custom error page
            $this->show_payment_error('Order not found. Please contact support if you were charged.');
            exit;
        }
    
        $status = $data['payment_status'] ?? '';
        $this->process_payment_status($order, $status, $data['transaction_ref'] ?? '', 'return URL');
    
        switch ($status) {
            case 'completed':
            case 'pending':
                // Success - redirect to order received page
                wp_redirect($order->get_checkout_order_received_url());
                break;
            
            case 'failed':
                // Restore cart and show error
                $this->restore_order_to_cart($order);
                wc_add_notice(__('Payment failed. Please try again or use a different payment method.', 'woocommerce-nexzoneo'), 'error');
                wp_redirect(wc_get_checkout_url());
                break;
            
            case 'cancelled':
                // Restore cart and show notice
                $this->restore_order_to_cart($order);
                wc_add_notice(__('Payment was cancelled. Your order is still in your cart.', 'woocommerce-nexzoneo'), 'notice');
                wp_redirect(wc_get_checkout_url());
                break;
            
            case 'expired':
                // Restore cart and show error
                $this->restore_order_to_cart($order);
                wc_add_notice(__('Payment session expired. Please try again.', 'woocommerce-nexzoneo'), 'error');
                wp_redirect(wc_get_checkout_url());
                break;
            
            case 'partially_refunded':
            case 'fully_refunded':
                // Refunded orders - redirect to order received page
                wp_redirect($order->get_checkout_order_received_url());
                break;
            
            default:
                // Unknown status - restore cart and show warning
                $this->restore_order_to_cart($order);
                wc_add_notice(__('Payment status unknown. Please contact support if funds were deducted from your account.', 'woocommerce-nexzoneo'), 'error');
                wp_redirect(wc_get_checkout_url());
                break;
        }
        exit;
    }

    /**
     * Handle TransBridge GET redirect return with enhanced security
     */
    private function handle_transbridge_return() {
        // Get order ID and security token from URL parameters
        $order_id = $_GET['order_id'] ?? $_GET['order'] ?? '';
        $session_token = $_GET['session_token'] ?? '';
        
        if (empty($order_id)) {
            // Critical error - show custom error page
            $this->show_payment_error('Unable to identify your order. Please contact support if payment was processed.');
            exit;
        }
        
        $order = wc_get_order($order_id);
        if (!$order) {
            // Critical error - show custom error page
            $this->show_payment_error('Order not found. Please contact support if payment was processed.');
            exit;
        }
        
        // Check order status after webhook processing
        $order_status = $order->get_status();
        
        if ($order->is_paid() || $order_status === 'processing' || $order_status === 'completed') {
            // Payment successful - redirect to success page
            wp_redirect($order->get_checkout_order_received_url());
            exit;
        }
        
        if (in_array($order_status, ['failed', 'cancelled', 'refunded'])) {
            // Payment failed - restore cart and redirect to checkout with error
            $this->restore_order_to_cart($order);
            
            if ($order_status === 'cancelled') {
                wc_add_notice(__('Payment was cancelled. Your order is still in your cart.', 'woocommerce-nexzoneo'), 'notice');
            } else {
                wc_add_notice(__('Payment was not successful. Please try again or use a different payment method.', 'woocommerce-nexzoneo'), 'error');
            }
            
            wp_redirect(wc_get_checkout_url());
            exit;
        }
        
        // Order still pending - check Nexzoneo payment status directly as fallback
        // This handles the timing issue where webhook hasn't been processed yet
        $nexzoneo_status = $this->check_nexzoneo_payment_status($order_id);
        
        if ($nexzoneo_status === 'completed') {
            // Payment is completed in Nexzoneo but WooCommerce order not updated yet
            // Update the order status and redirect to success
            $this->process_payment_status($order, 'completed', '', 'direct status check');
            wp_redirect($order->get_checkout_order_received_url());
            exit;
        }
        
        if (in_array($nexzoneo_status, ['failed', 'cancelled', 'expired'])) {
            // Payment failed in Nexzoneo - restore cart and redirect to checkout
            $this->restore_order_to_cart($order);
            
            if ($nexzoneo_status === 'cancelled') {
                wc_add_notice(__('Payment was cancelled. Your order is still in your cart.', 'woocommerce-nexzoneo'), 'notice');
            } elseif ($nexzoneo_status === 'expired') {
                wc_add_notice(__('Payment session expired. Please try again.', 'woocommerce-nexzoneo'), 'error');
            } else {
                wc_add_notice(__('Payment was not successful. Please try again or use a different payment method.', 'woocommerce-nexzoneo'), 'error');
            }
            
            wp_redirect(wc_get_checkout_url());
            exit;
        }
        
        // Both WooCommerce and Nexzoneo show pending - show processing message
        $this->show_payment_processing($order);
        exit;
    }

    /**
     * Show payment processing page
     */
    private function show_payment_processing($order) {
        $order_received_url = $order->get_checkout_order_received_url();
        $order_created = $order->get_date_created();
        $minutes_elapsed = (time() - $order_created->getTimestamp()) / 60;
        
        // If more than 3 minutes have passed, assume payment is successful and redirect
        if ($minutes_elapsed > 3) {
            wp_redirect($order_received_url);
            exit;
        }
        
        echo '<!DOCTYPE html>';
        echo '<html>';
        echo '<head>';
        echo '<title>Processing Payment</title>';
        echo '<meta charset="utf-8">';
        echo '<meta name="viewport" content="width=device-width, initial-scale=1">';
        echo '<style>';
        echo 'body { font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif; margin: 0; padding: 40px 20px; background: #f7f7f7; text-align: center; }';
        echo '.container { max-width: 500px; margin: 0 auto; background: white; padding: 40px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }';
        echo '.spinner { border: 4px solid #f3f3f3; border-top: 4px solid #0073aa; border-radius: 50%; width: 50px; height: 50px; animation: spin 1s linear infinite; margin: 20px auto; }';
        echo '@keyframes spin { 0% { transform: rotate(0deg); } 100% { transform: rotate(360deg); } }';
        echo '.btn { display: inline-block; padding: 12px 24px; background: #0073aa; color: white; text-decoration: none; border-radius: 4px; margin-top: 20px; }';
        echo '.btn:hover { background: #005a87; }';
        echo '.progress { margin: 20px 0; }';
        echo '.progress-text { font-size: 14px; color: #666; }';
        echo '</style>';
        echo '<script>';
        echo 'var refreshCount = 0;';
        echo 'var maxRefreshes = 36;'; // 3 minutes with 5-second intervals
        echo 'var orderReceivedUrl = "' . esc_js($order_received_url) . '";';
        echo 'function checkPaymentStatus() {';
        echo '    refreshCount++;';
        echo '    var progressText = document.getElementById("progressText");';
        echo '    progressText.innerHTML = "Checking payment status...";';
        echo '    if (refreshCount >= maxRefreshes) {';
        echo '        progressText.innerHTML = "Payment processing complete. Redirecting...";';
        echo '        setTimeout(function() { window.location.href = orderReceivedUrl; }, 2000);';
        echo '        return;';
        echo '    }';
        echo '    setTimeout(function() { window.location.reload(); }, 5000);';
        echo '}';
        echo 'setTimeout(checkPaymentStatus, 5000);';
        echo '</script>';
        echo '</head>';
        echo '<body>';
        echo '<div class="container">';
        echo '<h2>Processing Your Payment</h2>';
        echo '<div class="spinner"></div>';
        echo '<p>Your payment is being processed. You will be redirected automatically.</p>';
        echo '<p><strong>Order #' . $order->get_order_number() . '</strong></p>';
        echo '<div class="progress">';
        echo '<p id="progressText" class="progress-text">Checking payment status...</p>';
        echo '</div>';
        echo '<a href="' . esc_url($order_received_url) . '" class="btn">Continue to Order Confirmation</a>';
        echo '</div>';
        echo '</body>';
        echo '</html>';
    }

    /**
     * Restore order items back to cart
     * This is used when payment fails/cancelled so customer can try again
     */
    private function restore_order_to_cart($order) {
        if (!$order) {
            return false;
        }
        
        // Check if cart is already populated (to avoid duplicates)
        if (!WC()->cart->is_empty()) {
            // Don't restore if cart already has items
            return false;
        }
        
        try {
            // Get order items
            $order_items = $order->get_items();
            
            if (empty($order_items)) {
                return false;
            }
            
            // Add each item back to cart
            foreach ($order_items as $item) {
                $product_id = $item->get_product_id();
                $quantity = $item->get_quantity();
                $variation_id = $item->get_variation_id();
                $variation = $item->get_meta_data();
                
                // Build variation data array
                $variation_data = array();
                if ($variation_id) {
                    foreach ($variation as $meta) {
                        $data = $meta->get_data();
                        if (strpos($data['key'], 'pa_') === 0 || strpos($data['key'], 'attribute_') === 0) {
                            $variation_data[$data['key']] = $data['value'];
                        }
                    }
                }
                
                // Add to cart
                if ($variation_id) {
                    WC()->cart->add_to_cart($product_id, $quantity, $variation_id, $variation_data);
                } else {
                    WC()->cart->add_to_cart($product_id, $quantity);
                }
            }
            
            // Restore any coupons that were applied
            $coupons = $order->get_coupon_codes();
            if (!empty($coupons)) {
                foreach ($coupons as $coupon_code) {
                    WC()->cart->apply_coupon($coupon_code);
                }
            }
            
            return true;
            
        } catch (Exception $e) {
            // Log error but don't fail the entire flow
            if ($this->debug === 'yes') {
                $this->log('Cart restoration failed: ' . $e->getMessage());
            }
            return false;
        }
    }

    /**
     * Display payment error page (only for critical errors)
     * For user-facing errors, use WooCommerce notices instead
     */
    private function show_payment_error($error_message = '') {
        $message = $error_message ?: 'Payment error occurred';
        $checkout_url = wc_get_checkout_url();
        
        echo '<!DOCTYPE html>
<html>
<head>
    <title>Payment Error</title>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <style>
        body { font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif; margin: 0; padding: 40px 20px; background: #f7f7f7; text-align: center; }
        .container { max-width: 500px; margin: 0 auto; background: white; padding: 40px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .error-icon { font-size: 48px; color: #e74c3c; margin-bottom: 20px; }
        .btn { display: inline-block; padding: 12px 24px; background: #0073aa; color: white; text-decoration: none; border-radius: 4px; margin-top: 20px; }
        .btn:hover { background: #005a87; }
    </style>
</head>
<body>
    <div class="container">
        <div class="error-icon">⚠️</div>
        <h1>Payment Error</h1>
        <p>' . esc_html($message) . '</p>
        <a href="' . esc_url($checkout_url) . '" class="btn">Return to Checkout</a>
    </div>
</body>
</html>';
        exit;
    }

    /**
     * Prepare data for signature verification
     */
    private function prepare_verify_data($data) {
        $verify_data = [
            'order_id' => $data['order_id'] ?? '',
            'total' => number_format((float)($data['total'] ?? 0), 8, '.', ''),
            'currency' => $data['currency'] ?? '',
            'payment_status' => $data['payment_status'] ?? '',
            'session_status' => $data['session_status'] ?? '',
            'transaction_ref' => $data['transaction_ref'] ?? '',
            'customer_email' => $data['customer_email'] ?? '',
            'timestamp' => $data['timestamp'] ?? ''
        ];
        
        $verify_data = array_filter($verify_data);
        ksort($verify_data);
        return $verify_data;
    }

    /**
     * Verify signature of incoming data
     */
    private function verify_signature($verify_data, $signature) {
        $settings = get_option('woocommerce_nexzoneo_settings', []);
        $security_token = $settings['security_token'] ?? '';
        
        $expected_signature = hash_hmac('sha256', json_encode($verify_data), $security_token);
        return hash_equals($expected_signature, $signature);
    }

    /**
     * Process payment status update
     */
    private function process_payment_status($order, $status, $transaction_ref = '', $source = '') {
        if (!$order) {
            return false;
        }

        switch ($status) {
            case 'completed':
                $order->payment_complete($transaction_ref);
                $order->add_order_note(sprintf('Payment completed via %s', $source));
                break;
                
            case 'failed':
                $order->update_status('failed', sprintf('Payment failed via %s', $source));
                break;

            case 'cancelled':
                $order->update_status('cancelled', sprintf('Payment cancelled via %s', $source));
                break;
                
            case 'expired':
                $order->update_status('cancelled', sprintf('Payment expired via %s', $source));
                $order->add_order_note('Payment session expired - order automatically cancelled');
                break;
                
            default:
                error_log("Nexzoneo Pay: Unknown payment status '{$status}' for order {$order->get_id()}");
                $order->add_order_note(sprintf('Unknown payment status (%s) received via %s - treated as failed', $status, $source));
                $order->update_status('failed', sprintf('Unknown payment status (%s) via %s', $status, $source));
                break;
        }

        $order->save();
        return true;
    }
    
    /**
     * Handle webhook
     */
    public function handle_webhook() {
        try {
            $this->log('Webhook received from Nexzoneo');

            // Get raw payload
            $payload = file_get_contents('php://input');
            if (empty($payload)) {
                $this->log('Empty webhook payload received', 'error');
                status_header(400);
                exit('Empty payload');
            }

            // Validate webhook signature if secret is configured
            if (!empty($this->webhook_secret)) {
                $signature = $_SERVER['HTTP_X_NEXZONEO_SIGNATURE'] ?? '';
                if (!$this->validate_webhook_signature($payload, $signature)) {
                    $this->log('Invalid webhook signature', 'error');
                    status_header(401);
                    exit('Invalid signature');
                }
            }

            // Parse webhook data
            $webhook_data = json_decode($payload, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                $this->log('Invalid JSON in webhook payload', 'error');
                status_header(400);
                exit('Invalid JSON');
            }

            $this->log('Webhook data: ' . wp_json_encode($webhook_data));

            // Process webhook based on event type
            $this->process_webhook_event($webhook_data);

            status_header(200);
            exit('OK');

        } catch (Exception $e) {
            $this->log('Webhook processing error: ' . $e->getMessage(), 'error');
            status_header(500);
            exit('Webhook processing failed');
        }
    }

    /**
     * Process webhook event with enhanced handling
     */
    private function process_webhook_event($webhook_data) {
        $event_type = $webhook_data['event_type'] ?? '';
        $session_id = $webhook_data['session_id'] ?? '';
        $order_id = $webhook_data['order_id'] ?? '';

        $this->log("Processing webhook event: {$event_type} for session: {$session_id}, order: {$order_id}");

        // Find order by session ID or order ID
        $order = $this->find_order_by_webhook_data($webhook_data);
        if (!$order) {
            $this->log('Order not found for webhook data', 'error');
            return;
        }

        // Update transaction record
        $this->update_transaction_record($order, $webhook_data);

        // Process based on event type
        switch ($event_type) {
            case 'payment.completed':
            case 'payment_intent.succeeded':
                $this->handle_payment_completed($order, $webhook_data);
                break;
                
            case 'payment.failed':
            case 'payment_intent.payment_failed':
                $this->handle_payment_failed($order, $webhook_data);
                break;
                
            case 'payment.cancelled':
            case 'payment_intent.canceled':
                $this->handle_payment_cancelled($order, $webhook_data);
                break;
                
            case 'refund.completed':
            case 'charge.refunded':
                $this->handle_refund_completed($order, $webhook_data);
                break;
                
            case 'dispute.created':
            case 'chargeback.created':
                $this->handle_dispute_created($order, $webhook_data);
                break;
                
            case 'dispute.won':
            case 'chargeback.won':
                $this->handle_dispute_won($order, $webhook_data);
                break;
                
            case 'dispute.lost':
            case 'chargeback.lost':
                $this->handle_dispute_lost($order, $webhook_data);
                break;
                
            default:
                $this->log("Unhandled webhook event type: {$event_type}");
                break;
        }
    }

    /**
     * Handle dispute created webhook
     */
    private function handle_dispute_created($order, $webhook_data) {
        $this->log("Processing dispute created for order {$order->get_id()}");
        
        // Add order note about dispute
        $dispute_amount = $webhook_data['amount'] ?? $order->get_total();
        $dispute_reason = $webhook_data['reason'] ?? 'Unknown';
        
        $order->add_order_note(
            sprintf(
                'Payment disputed: %s %s (Reason: %s). Dispute ID: %s',
                $dispute_amount,
                $order->get_currency(),
                $dispute_reason,
                $webhook_data['dispute_id'] ?? 'N/A'
            )
        );
        
        // Update order status to on-hold for review
        $order->update_status('on-hold', 'Payment disputed - under review');
        
        // Send email notification to admin
        $this->send_dispute_notification($order, $webhook_data, 'created');
    }

    /**
     * Handle dispute won webhook
     */
    private function handle_dispute_won($order, $webhook_data) {
        $this->log("Processing dispute won for order {$order->get_id()}");
        
        // Add order note about dispute resolution
        $order->add_order_note(
            sprintf(
                'Dispute won! Payment dispute resolved in merchant favor. Dispute ID: %s',
                $webhook_data['dispute_id'] ?? 'N/A'
            )
        );
        
        // Restore order to processing status if it was on hold
        if ($order->get_status() === 'on-hold') {
            $order->update_status('processing', 'Dispute resolved - payment confirmed');
        }
        
        // Send email notification
        $this->send_dispute_notification($order, $webhook_data, 'won');
    }

    /**
     * Handle dispute lost webhook
     */
    private function handle_dispute_lost($order, $webhook_data) {
        $this->log("Processing dispute lost for order {$order->get_id()}");
        
        // Add order note about dispute resolution
        $dispute_amount = $webhook_data['amount'] ?? $order->get_total();
        $order->add_order_note(
            sprintf(
                'Dispute lost: %s %s has been charged back. Dispute ID: %s',
                $dispute_amount,
                $order->get_currency(),
                $webhook_data['dispute_id'] ?? 'N/A'
            )
        );
        
        // Update order status to cancelled or refunded
        $order->update_status('cancelled', 'Payment dispute lost - funds charged back');
        
        // Send email notification
        $this->send_dispute_notification($order, $webhook_data, 'lost');
    }

    /**
     * Send dispute notification email
     */
    private function send_dispute_notification($order, $webhook_data, $status) {
        $admin_email = get_option('admin_email');
        $site_name = get_bloginfo('name');
        
        $subject = sprintf('[%s] Payment Dispute %s - Order #%s', 
            $site_name, 
            ucfirst($status), 
            $order->get_order_number()
        );
        
        $message = sprintf(
            "A payment dispute has been %s for order #%s.\n\n" .
            "Order Details:\n" .
            "- Order ID: %s\n" .
            "- Amount: %s %s\n" .
            "- Customer: %s\n" .
            "- Dispute ID: %s\n" .
            "- Reason: %s\n\n" .
            "View order: %s\n\n" .
            "Manage disputes in your Nexzoneo dashboard: https://payment.nexzoneo.com/cp/",
            $status,
            $order->get_order_number(),
            $order->get_id(),
            $order->get_total(),
            $order->get_currency(),
            $order->get_billing_email(),
            $webhook_data['dispute_id'] ?? 'N/A',
            $webhook_data['reason'] ?? 'Unknown',
            $order->get_edit_order_url()
        );
        
        wp_mail($admin_email, $subject, $message);
    }

    /**
     * Process refunds
     */
    public function process_refund($order_id, $amount = null, $reason = '') {
        $order = wc_get_order($order_id);
        
        if (!$order) {
            return new WP_Error('invalid_order', 'Order not found');
        }
        
        try {
            // Create API instance
            $api = new Nexzoneo_API(
                $this->site_key,
                $this->security_token,
                $this->environment,
                $this->debug === 'yes'
            );
            
            // Get transaction data
            $transaction_data = $this->get_transaction_data($order);
            if (!$transaction_data) {
                return new WP_Error('no_transaction', 'No transaction data found for this order');
            }
            
            // Prepare refund data
            $refund_data = array(
                'session_token' => $transaction_data['session_token'],
                'amount' => $amount ?: $order->get_total(),
                'reason' => $reason ?: 'Refund requested'
            );
            
            // Create refund
            $response = $api->create_refund($refund_data);
            
            if ($response['success']) {
                $order->add_order_note(
                    sprintf('Refund of %s processed successfully. Refund ID: %s', 
                        wc_price($amount), 
                        $response['refund_id'] ?? 'N/A'
                    )
                );
                return true;
            } else {
                return new WP_Error('refund_failed', $response['error'] ?? 'Refund processing failed');
            }
            
        } catch (Exception $e) {
            $this->log('Refund processing error: ' . $e->getMessage(), 'error');
            return new WP_Error('refund_error', $e->getMessage());
        }
    }
    
    /**
     * Helper methods
     */
    private function get_order_items($order) {
        $items = array();
        foreach ($order->get_items() as $item) {
            $items[] = array(
                'name' => $item->get_name(),
                'quantity' => $item->get_quantity(),
                'price' => $item->get_total()
            );
        }
        return $items;
    }
    
    private function store_transaction_data($order, $response) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'nexzoneo_transactions';
        
        $wpdb->insert(
            $table_name,
            array(
                'order_id' => $order->get_id(),
                'session_token' => $response['session_token'],
                'amount' => $order->get_total(),
                'currency' => $order->get_currency(),
                'status' => 'pending'
            ),
            array('%d', '%s', '%f', '%s', '%s')
        );
    }
    
    private function get_transaction_data($order) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'nexzoneo_transactions';
        
        return $wpdb->get_row(
            $wpdb->prepare("SELECT * FROM $table_name WHERE order_id = %d", $order->get_id()),
            ARRAY_A
        );
    }
    
    private function find_order_by_webhook_data($webhook_data) {
        // Implementation for finding order by webhook data
        $order_id = $webhook_data['order_id'] ?? '';
        if ($order_id) {
            return wc_get_order($order_id);
        }
        return null;
    }
    
    private function update_transaction_record($order, $webhook_data) {
        // Implementation for updating transaction record
    }
    
    private function handle_payment_completed($order, $webhook_data) {
        $order->payment_complete();
        $order->add_order_note('Payment completed via Nexzoneo');
    }
    
    private function handle_payment_failed($order, $webhook_data) {
        $order->update_status('failed', 'Payment failed');
    }
    
    private function handle_payment_cancelled($order, $webhook_data) {
        $order->update_status('cancelled', 'Payment cancelled');
    }
    
    private function handle_refund_completed($order, $webhook_data) {
        $order->add_order_note('Refund completed via Nexzoneo');
    }
    
    private function validate_webhook_signature($payload, $signature) {
        if (empty($this->webhook_secret)) {
            return true;
        }
        
        $expected_signature = hash_hmac('sha256', $payload, $this->webhook_secret);
        return hash_equals($expected_signature, $signature);
    }
    
    private function log($message, $level = 'info') {
        if ($this->debug === 'yes' && function_exists('wc_get_logger')) {
            $logger = wc_get_logger();
            $logger->log($level, $message, array('source' => 'nexzoneo'));
        }
    }
    
    /**
     * Check Nexzoneo payment status directly
     * This is used as a fallback when webhook processing is delayed
     */
    private function check_nexzoneo_payment_status($order_id) {
        try {
            // Get the Nexzoneo API endpoint
            $api_base = 'https://payment.nexzoneo.com/api/merchant';
            
            // Get site credentials (map WooCommerce field names to API parameters)
            $site_key = $this->get_option('site_key');
            $security_token = $this->get_option('security_token');
            
            if (empty($site_key) || empty($security_token)) {
                return 'unknown';
            }
            
            // Build API URL to check payment status by order ID
            $api_url = $api_base . '/payment-status';
            
            // Prepare request data (only order_id in POST body)
            $request_data = [
                'order_id' => $order_id
            ];
            
            // Make API request using cURL with proper authentication headers
            $ch = curl_init();
            curl_setopt_array($ch, [
                CURLOPT_URL => $api_url,
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => json_encode($request_data),
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TIMEOUT => 10,
                CURLOPT_HTTPHEADER => [
                    'Content-Type: application/json',
                    'X-Site-Key: ' . $site_key,
                    'Authorization: Bearer ' . $security_token,
                    'User-Agent: WooCommerce-Nexzoneo-Plugin/1.0.34'
                ]
            ]);
            
            $response = curl_exec($ch);
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);
            
            if ($error) {
                return 'unknown';
            }
            
            if ($http_code !== 200) {
                return 'unknown';
            }
            
            $data = json_decode($response, true);
            
            if (!$data || !isset($data['status'])) {
                return 'unknown';
            }
            
            if ($data['status'] !== 'success') {
                return 'unknown';
            }
            
            // Map Nexzoneo status to WooCommerce-compatible status
            // Handle double-nested response structure
            $nexzoneo_status = $data['data']['data']['current_status'] ?? $data['data']['current_status'] ?? 0;
            
            switch ((int)$nexzoneo_status) {
                case 2: // Completed
                    return 'completed';
                case 3: // Failed
                    return 'failed';
                case 4: // Expired
                    return 'expired';
                case 5: // Cancelled
                    return 'cancelled';
                case 0: // Created
                case 1: // Processing
                default:
                    return 'pending';
            }
            
        } catch (Exception $e) {
            return 'unknown';
        }
    }

    /**
     * Admin options
     */
    public function admin_options() {
        ?>
        <h2><?php echo esc_html($this->get_method_title()); ?></h2>
        <p><?php echo esc_html($this->get_method_description()); ?></p>
        
        <div class="nexzoneo-admin-notice" style="background: #f0f8ff; border: 1px solid #0073aa; padding: 15px; margin: 20px 0; border-radius: 4px;">
            <h4 style="margin-top: 0;">🚀 Enhanced Version 1.0.30</h4>
            <p><strong>Latest Fixes:</strong></p>
            <ul>
                <li>✅ <strong>FIXED:</strong> WooCommerce Blocks compatibility issue causing "no payment methods available" error</li>
                <li>✅ Added proper cart_checkout_blocks compatibility declaration</li>
                <li>✅ Fixed WooCommerce Blocks integration and registration</li>
                <li>✅ Enhanced JavaScript loading for block-based checkout</li>
                <li>✅ Improved payment method visibility in block editor</li>
                <li>✅ Maintains full backward compatibility with classic checkout</li>
            </ul>
            <p><strong>Webhook URL:</strong> <code><?php echo home_url('/wc-api/nexzoneo'); ?></code></p>
            <p><a href="https://payment.nexzoneo.com/cp/" target="_blank">Open Nexzoneo Dashboard</a> | 
               <a href="https://docs.nexzoneo.com/woocommerce/" target="_blank">Documentation</a> |
               <a href="https://nexzoneo.freshdesk.com" target="_blank">Support</a></p>
        </div>

        <table class="form-table">
            <?php $this->generate_settings_html(); ?>
        </table>
        <?php
    }
}
?>
