<?php
/**
 * Plugin Name: WooCommerce Gateway Nexzoneo Pay
 * Plugin URI: https://payment.nexzoneo.com/plugins/woocommerce
 * Description: Accept payments via Nexzoneo Pay in your WooCommerce store with enhanced reliability and performance
 * Version: 1.0.37
 * Author: Nexzoneo
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * WC requires at least: 5.0
 * WC tested up to: 10.1.2
 * Woo: 12345:a2c5d1e83f71d
 * WooCommerce HPOS Compatible: true
 * 
 * Changelog:
 * 1.0.37:
 * - IMPROVED: Payment error handling now uses native WooCommerce notices
 * - ENHANCED: Better UX - users stay within themed checkout page for errors
 * - ADDED: Automatic cart restoration when payment fails/cancelled/expires
 * - IMPROVED: Professional error/notice messages following WooCommerce conventions
 * - FIXED: Custom error page now only used for critical security errors
 * - UX: Users can immediately retry payment without manual cart restoration
 * - CONSISTENCY: All error handling now matches WooCommerce and major payment gateways
 *
 * 1.0.36:
 * - IMPROVED: WordPress Site Health compatibility and recommendations
 * - ENHANCED: Better debug logging practices and security recommendations
 * - UPDATED: Documentation for WordPress best practices compliance
 * - OPTIMIZED: Plugin performance and resource usage
 * - ADDED: Enhanced error handling for edge cases
 * - SECURITY: Improved validation and sanitization throughout
 * - COMPATIBILITY: Tested with WordPress 6.6 and PHP 8.1+
 * - DOCUMENTATION: Expanded readme with best practices and troubleshooting
 *
 * 1.0.35:
 * - ADDED: Customer phone number now sent to payment API (required for Swipelux and other providers)
 * - ADDED: Customer first name and last name separately sent for better provider compatibility
 * - ENHANCED: Full customer data now transmitted to support all payment provider requirements
 * - IMPROVED: Better compliance with payment provider KYC/AML requirements
 *
 * 1.0.34:
 * - FIXED: Critical syntax error in class-nexzoneo-api.php that caused WordPress fatal errors
 * - CLEANED: Removed all debug logging statements for production stability
 * - VERIFIED: All PHP files pass syntax validation for reliable operation
 * - STABLE: Maintains all functionality from v1.0.33 with improved reliability
 *
 * 1.0.32:
 * - CRITICAL FIX: Comprehensive error message sanitization across entire payment system
 * - FIXED: Provider names (Transbridge, Stripe) completely removed from all user-facing errors
 * - ENHANCED: Error sanitization now applied at the source (provider classes) and session level
 * - SECURITY: Full compliance with payment provider visibility policies throughout system
 * - IMPROVED: Professional error messages that never expose internal payment provider details
 *
 * 1.0.31:
 * - ENHANCED: Sanitized all error messages to remove payment provider names from customer view
 * - SECURITY: Removed all references to Transbridge, Stripe, and other providers from user-facing errors
 * - IMPROVED: Professional error messages that provide clear guidance without exposing technical details
 * - FIXED: Error message chains now display clean, user-friendly text instead of technical API errors
 * - COMPLIANCE: Full adherence to payment provider visibility policies for enhanced security
 *
 * 1.0.30:
 * - FIXED: WooCommerce Blocks compatibility issue causing "no payment methods available" error
 * - Added proper cart_checkout_blocks compatibility declaration
 * - Fixed WooCommerce Blocks integration and registration
 * - Enhanced JavaScript loading for block-based checkout
 * - Improved payment method visibility in block editor
 * 
 * 1.0.29:
 * - Enhanced dispute management integration with new dispute handling system
 * - Improved settlement processing with dispute reserves and fee calculations
 * - Advanced webhook validation and processing reliability
 * - Better payment reconciliation and monitoring capabilities
 * - Enhanced security with improved fraud protection
 * - Updated compatibility with latest Nexzoneo payment system (September 2025)
 * - Improved error handling and comprehensive logging
 * - Better support for refund processing and chargeback management
 */

defined('ABSPATH') || exit;

/**
 * Initialize the gateway after checking dependencies
 */
function init_nexzoneo_gateway() {
    if (!class_exists('WC_Payment_Gateway')) {
        return;
    }

    try {
        $api_file = __DIR__ . '/class-nexzoneo-api.php';
        $gateway_file = __DIR__ . '/class-nexzoneo-gateway.php';
        
        foreach ([$api_file, $gateway_file] as $file) {
            if (!file_exists($file)) {
                return;
            }
        }
        
        require_once $api_file;
        require_once $gateway_file;
    } catch (Exception $e) {
        return;
    }
}

/**
 * Add the gateway to WooCommerce
 */
function add_nexzoneo_gateway($methods) {
    $methods[] = 'WC_Gateway_Nexzoneo';
    return $methods;
}

/**
 * Plugin activation hook
 */
function nexzoneo_gateway_activate() {
    if (!class_exists('WooCommerce')) {
        deactivate_plugins(plugin_basename(__FILE__));
        wp_die('WooCommerce Gateway Nexzoneo requires WooCommerce to be installed and active.');
    }
    
    // Create transaction table
    global $wpdb;
    $table_name = $wpdb->prefix . 'nexzoneo_transactions';
    
    $charset_collate = $wpdb->get_charset_collate();
    
    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        order_id bigint(20) NOT NULL,
        session_token varchar(255) NOT NULL,
        payment_intent_id varchar(255) DEFAULT NULL,
        transaction_id varchar(255) DEFAULT NULL,
        status varchar(50) DEFAULT 'pending',
        amount decimal(10,2) NOT NULL,
        currency varchar(3) NOT NULL,
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        UNIQUE KEY order_id (order_id),
        KEY session_token (session_token),
        KEY payment_intent_id (payment_intent_id),
        KEY status (status)
    ) $charset_collate;";
    
    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
}

/**
 * Plugin deactivation hook
 */
function nexzoneo_gateway_deactivate() {
    // Clean up any scheduled events if needed
    wp_clear_scheduled_hook('nexzoneo_cleanup_transactions');
}

/**
 * Add custom order status for Nexzoneo payments
 */
function nexzoneo_register_order_status() {
    register_post_status('wc-nexzoneo-pending', array(
        'label' => 'Nexzoneo Pending',
        'public' => true,
        'exclude_from_search' => false,
        'show_in_admin_all_list' => true,
        'show_in_admin_status_list' => true,
        'label_count' => _n_noop('Nexzoneo Pending <span class="count">(%s)</span>', 'Nexzoneo Pending <span class="count">(%s)</span>')
    ));
}

/**
 * Add custom order status to WooCommerce status list
 */
function nexzoneo_add_order_status($order_statuses) {
    $new_order_statuses = array();
    
    foreach ($order_statuses as $key => $status) {
        $new_order_statuses[$key] = $status;
        
        if ('wc-pending' === $key) {
            $new_order_statuses['wc-nexzoneo-pending'] = 'Nexzoneo Pending';
        }
    }
    
    return $new_order_statuses;
}

/**
 * Add admin notice for missing WooCommerce
 */
function nexzoneo_woocommerce_missing_notice() {
    echo '<div class="error"><p><strong>WooCommerce Gateway Nexzoneo</strong> requires WooCommerce to be installed and active.</p></div>';
}

/**
 * Check if WooCommerce is active and initialize
 */
function nexzoneo_gateway_init() {
    if (!class_exists('WooCommerce')) {
        add_action('admin_notices', 'nexzoneo_woocommerce_missing_notice');
        return;
    }
    
    // Initialize the gateway
    add_action('plugins_loaded', 'init_nexzoneo_gateway', 11);
    add_filter('woocommerce_payment_gateways', 'add_nexzoneo_gateway');
    
    // Register custom order status
    add_action('init', 'nexzoneo_register_order_status');
    add_filter('wc_order_statuses', 'nexzoneo_add_order_status');
}

// Plugin hooks
register_activation_hook(__FILE__, 'nexzoneo_gateway_activate');
register_deactivation_hook(__FILE__, 'nexzoneo_gateway_deactivate');

// Initialize
add_action('plugins_loaded', 'nexzoneo_gateway_init');

/**
 * Add settings link to plugin page
 */
function nexzoneo_plugin_action_links($links) {
    $settings_link = '<a href="' . admin_url('admin.php?page=wc-settings&tab=checkout&section=nexzoneo') . '">Settings</a>';
    array_unshift($links, $settings_link);
    return $links;
}
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'nexzoneo_plugin_action_links');

/**
 * Add support for WooCommerce HPOS (High-Performance Order Storage) and Blocks
 */
function nexzoneo_declare_compatibility() {
    if (class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil')) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('cart_checkout_blocks', __FILE__, true);
    }
}
add_action('before_woocommerce_init', 'nexzoneo_declare_compatibility');

/**
 * Load plugin textdomain for translations
 */
function nexzoneo_load_textdomain() {
    load_plugin_textdomain('woocommerce-gateway-nexzoneo', false, dirname(plugin_basename(__FILE__)) . '/languages/');
}
add_action('plugins_loaded', 'nexzoneo_load_textdomain');

/**
 * Initialize WooCommerce Blocks integration
 */
function init_nexzoneo_blocks_support() {
    if (!class_exists('Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType')) {
        return;
    }

    try {
        class WC_Nexzoneo_Blocks_Support extends Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType {
            protected $name = 'nexzoneo';

            public function initialize() {
                $this->settings = get_option('woocommerce_nexzoneo_settings', array());
            }

            public function is_active() {
                return !empty($this->settings['enabled']) && 'yes' === $this->settings['enabled'];
            }

            public function get_payment_method_script_handles() {
                wp_register_script(
                    'nexzoneo-blocks-integration',
                    plugins_url('nexzoneo-blocks.js', __FILE__),
                    array('wc-blocks-registry'),
                    '1.0.37',
                    true
                );
                return array('nexzoneo-blocks-integration');
            }

            public function get_payment_method_data() {
                return array(
                    'title' => $this->settings['title'] ?? 'Credit Card / Debit Card',
                    'description' => $this->settings['description'] ?? 'Pay securely with your credit card, debit card, or bank transfer via Nexzoneo',
                    'supports' => array('products'),
                );
            }
        }

        add_action(
            'woocommerce_blocks_payment_method_type_registration',
            function($registry) {
                $registry->register(new WC_Nexzoneo_Blocks_Support());
            }
        );

    } catch (Exception $e) {}
}
add_action('woocommerce_after_register_post_type', 'init_nexzoneo_blocks_support');

/**
 * Enqueue and localize scripts for checkout
 */
add_action('wp_enqueue_scripts', function() {
    if (! is_checkout()) {
        return;
    }
    
    $settings = get_option('woocommerce_nexzoneo_settings', []);
    wp_localize_script('nexzoneo-blocks-integration', 'nexzoneoSettings', [
        'enabled' => $settings['enabled'] ?? 'no',
        'hasCredentials' => (!empty($settings['site_key']) && !empty($settings['security_token'])) ? 'yes' : 'no',
        'description' => $settings['description'] ?? ''
    ]);
});

/**
 * Enhanced error logging for debugging
 */
function nexzoneo_log($message, $level = 'info') {
    if (class_exists('WC_Logger')) {
        $logger = wc_get_logger();
        $logger->log($level, $message, array('source' => 'nexzoneo'));
    }
}

/**
 * Cleanup old transaction records (run daily)
 */
function nexzoneo_cleanup_old_transactions() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'nexzoneo_transactions';
    
    // Delete transactions older than 30 days that are still pending
    $wpdb->query($wpdb->prepare(
        "DELETE FROM $table_name WHERE status = 'pending' AND created_at < DATE_SUB(NOW(), INTERVAL 30 DAY)"
    ));
}

// Schedule cleanup if not already scheduled
if (!wp_next_scheduled('nexzoneo_cleanup_transactions')) {
    wp_schedule_event(time(), 'daily', 'nexzoneo_cleanup_transactions');
}
add_action('nexzoneo_cleanup_transactions', 'nexzoneo_cleanup_old_transactions');
